function time = measureEdgeTiming2(waveform,threshold,hysteresis,sampleInterval)
% This function finds the edges in a waveform.
% An edge is defined as the point where the waveform crosses the threshold
% The function includes a provision to define a guard band which can be used
% to reduce the sensitivity to noise.  The guard band is defined by the hysteresis
% values.
% The function will ensure that the waveform moves outside the guard band between
% successive edges.
% The edges are located by time.
% The first sample is considered as time zero.

% Modified by Kalev 5/2/2001

edgeNbr = 1;
thresholdTest = 0;
% threshold test is a state variable, its values have the following meanings
% (remember, it takes two samples to define an edge)
%    0 = looking for a value outside the guard band
%    1 = looking for a positive edge. a positive edge occurs when we find a sample
%        at or above the threshold 
%    2 = looking for a negative edge. a negative edge occurs when we find a sample
%        at or below the threshold 
%    3 = looking for a sample above the guard band before we look for a negative 
%        edge. it takes two samples to define an edge.  in this case the second 
%        sample that defined a positive edge was in the guard band.
%    4 = looking for a sample below the guard band before we look for a positive 
%        edge. it takes two samples to define an edge.  in this case the second 
%        sample that defined a negative edge was in the guard band.

% preallocate memory for time
% factor of 5 is based on reasonable sampling of at least 5 samples per symbol
% time = zeros(1, length(waveform)/5);
% this assumption is removed for higher precision
time = zeros(1, length(waveform));
 
for index = 1:length(waveform)-1; % iterate through the entire waveform
   switch thresholdTest;
   	case 0 % looking for a value outside the guard band
   	   if waveform(index) > threshold + hysteresis;
   	      thresholdTest = 2;
   	   elseif waveform(index) < threshold - hysteresis;
    	      thresholdTest = 1;
      	end
      
   	case 1 % below threshold, looking for positive threshold crossing
         if waveform(index) >= threshold;
            if abs(waveform(index) - waveform(index-1)) > eps
                fractional = (threshold - waveform(index-1))/(waveform(index) - waveform(index-1));
            else
                abs(waveform(index) - waveform(index-1))
                fractional = 0;
            end
				time(edgeNbr) = (index -1 + fractional)*sampleInterval;
           	edgeNbr = edgeNbr + 1;
            if waveform(index) > threshold + hysteresis;
               thresholdTest = 2;
            else
               thresholdTest = 3;
            end
         end
      
   	case 2 % above threshold looking for negative crossing
      	if waveform(index) <= threshold;
            if (waveform(index) - waveform(index-1)) ~= 0
                fractional = (threshold - waveform(index-1))/(waveform(index) - waveform(index-1));
            else
                fractional = 0;
            end
				time(edgeNbr) = (index - 1 + fractional)*sampleInterval;
         	edgeNbr = edgeNbr + 1;
            if waveform(index) < threshold - hysteresis;
               thresholdTest = 1;
            else
               thresholdTest = 4;
            end
         end
         
      case 3 % looking for a sample above the threshold + hysteresis
         if waveform(index) > threshold + hysteresis;
            thresholdTest = 2;
         end
         
      case 4 % looking for a sample below the threshold - hysteresis
         if waveform(index) < threshold - hysteresis;
            thresholdTest = 1;
         end 
   end
end
% deallocate unused elements of time
time = time(1:edgeNbr-1);
